'use client';

import React, { useState } from 'react';
import { motion, useMotionValue, useTransform } from 'framer-motion';
import { Sparkles, Target, Lightbulb, Shield, Eye, Compass, Leaf, Network } from 'lucide-react';

// Mock useLanguage hook for demo
const useLanguage = () => ({
  content: {
    aboutUs: {
      title: "About Us",
      text: "We are dedicated to transforming ideas into reality through innovative solutions and cutting-edge technology.",
      vision: {
        title: "Vision",
        text: "To be the leading force in digital transformation, empowering businesses to thrive in an ever-evolving technological landscape."
      },
      mission: {
        title: "Mission",
        text: "Delivering excellence through innovative solutions, fostering long-term partnerships, and creating sustainable value for our clients and communities."
      },
      values: {
        title: "Our Core Values",
        items: [
          {
            title: "Trust",
            text: "Building lasting relationships through transparency, reliability, and unwavering commitment to our clients' success."
          },
          {
            title: "Innovation",
            text: "Embracing creativity and cutting-edge technology to deliver solutions that push boundaries and exceed expectations."
          },
          {
            title: "Integration",
            text: "Creating seamless ecosystems where technology, people, and processes work in perfect harmony."
          },
          {
            title: "Sustainability",
            text: "Committed to responsible practices that ensure long-term success while protecting our planet for future generations."
          }
        ]
      }
    }
  }
});

const valueConfig = {
  Trust: {
    icon: Shield,
    gradient: "from-blue-600 via-cyan-500 to-teal-400",
    bgGradient: "from-blue-50/80 via-cyan-50/60 to-teal-50/40",
    glowColor: "rgba(59, 130, 246, 0.5)",
    pattern: "trust"
  },
  Innovation: {
    icon: Sparkles,
    gradient: "from-purple-600 via-pink-500 to-rose-400",
    bgGradient: "from-purple-50/80 via-pink-50/60 to-rose-50/40",
    glowColor: "rgba(168, 85, 247, 0.5)",
    pattern: "innovation"
  },
  Integration: {
    icon: Network,
    gradient: "from-emerald-600 via-green-500 to-lime-400",
    bgGradient: "from-emerald-50/80 via-green-50/60 to-lime-50/40",
    glowColor: "rgba(16, 185, 129, 0.5)",
    pattern: "integration"
  },
  Sustainability: {
    icon: Leaf,
    gradient: "from-orange-600 via-amber-500 to-yellow-400",
    bgGradient: "from-orange-50/80 via-amber-50/60 to-yellow-50/40",
    glowColor: "rgba(249, 115, 22, 0.5)",
    pattern: "sustainability"
  }
};

export default function AboutUsSection() {
  const { content } = useLanguage();
  const { aboutUs } = content;
  const [hoveredValue, setHoveredValue] = useState(null);

  return (
    <section id="about-us" className="scroll-mt-20 relative overflow-hidden bg-gradient-to-br from-slate-50 via-white to-blue-50/30">
      {/* Animated Background Elements */}
      <div className="absolute inset-0 overflow-hidden pointer-events-none">
        <motion.div
          className="absolute top-20 -left-20 w-96 h-96 bg-blue-400/10 rounded-full blur-3xl"
          animate={{
            x: [0, 100, 0],
            y: [0, -50, 0],
            scale: [1, 1.2, 1],
          }}
          transition={{ duration: 20, repeat: Infinity, ease: "easeInOut" }}
        />
        <motion.div
          className="absolute bottom-20 -right-20 w-96 h-96 bg-purple-400/10 rounded-full blur-3xl"
          animate={{
            x: [0, -100, 0],
            y: [0, 50, 0],
            scale: [1, 1.3, 1],
          }}
          transition={{ duration: 25, repeat: Infinity, ease: "easeInOut" }}
        />
        <motion.div
          className="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 w-[600px] h-[600px] bg-gradient-to-r from-cyan-400/5 to-pink-400/5 rounded-full blur-3xl"
          animate={{
            rotate: [0, 360],
            scale: [1, 1.1, 1],
          }}
          transition={{ duration: 30, repeat: Infinity, ease: "linear" }}
        />
      </div>

      <div className="relative z-10 px-4 py-20">
        {/* Header Section */}
        <motion.div 
          className="mx-auto max-w-2xl text-center mb-20"
          initial={{ opacity: 0, y: 20 }}
          whileInView={{ opacity: 1, y: 0 }}
          viewport={{ once: true }}
          transition={{ duration: 0.6 }}
        >
          <motion.div
            className="inline-flex items-center gap-2 px-4 py-2 bg-gradient-to-r from-blue-600/10 to-purple-600/10 rounded-full mb-6 backdrop-blur-sm border border-blue-200/50"
            initial={{ scale: 0 }}
            whileInView={{ scale: 1 }}
            viewport={{ once: true }}
            transition={{ type: "spring", duration: 0.8 }}
          >
            <Sparkles className="w-4 h-4 text-blue-600" />
            <span className="text-sm font-semibold bg-gradient-to-r from-blue-600 to-purple-600 bg-clip-text text-transparent">
              Who We Are
            </span>
          </motion.div>

          <h2 className="font-bold tracking-tight text-slate-900 text-4xl sm:text-5xl mb-6 relative">
            <motion.span
              className="inline-block"
              initial={{ opacity: 0, y: 20 }}
              whileInView={{ opacity: 1, y: 0 }}
              viewport={{ once: true }}
              transition={{ duration: 0.6, delay: 0.2 }}
            >
              {aboutUs.title}
            </motion.span>
            <motion.div
              className="absolute -bottom-2 left-1/2 -translate-x-1/2 h-1 bg-gradient-to-r from-blue-600 via-purple-600 to-pink-600 rounded-full"
              initial={{ width: 0 }}
              whileInView={{ width: "120px" }}
              viewport={{ once: true }}
              transition={{ duration: 0.8, delay: 0.4 }}
            />
          </h2>

          <p className="mt-8 text-lg leading-8 text-slate-600">
            {aboutUs.text}
          </p>
        </motion.div>

        <div className="max-w-7xl mx-auto">
          <div className="grid grid-cols-1 gap-12 md:grid-cols-2 lg:gap-16">
            {/* Vision & Mission */}
            <motion.div 
              className="space-y-8"
              initial={{ opacity: 0, x: -30 }}
              whileInView={{ opacity: 1, x: 0 }}
              viewport={{ once: true }}
              transition={{ duration: 0.7, delay: 0.2 }}
            >
              {/* Vision */}
              <motion.div
                className="relative group"
                whileHover={{ scale: 1.02 }}
                transition={{ type: "spring", stiffness: 300 }}
              >
                <div className="absolute inset-0 bg-gradient-to-br from-blue-600/20 to-cyan-600/20 rounded-3xl blur-xl opacity-0 group-hover:opacity-100 transition-opacity duration-500" />
                <div className="relative bg-white/80 backdrop-blur-sm rounded-3xl p-8 shadow-xl border border-slate-200/50 group-hover:border-blue-300/50 transition-all duration-500">
                  <div className="flex items-start gap-4">
                    <motion.div
                      className="flex-shrink-0 w-14 h-14 rounded-2xl bg-gradient-to-br from-blue-600 to-cyan-600 flex items-center justify-center shadow-lg"
                      whileHover={{ rotate: [0, -10, 10, -10, 0] }}
                      transition={{ duration: 0.5 }}
                    >
                      <Eye className="w-7 h-7 text-white" />
                    </motion.div>
                    <div className="flex-1">
                      <h3 className="text-2xl font-bold text-slate-900 mb-3 flex items-center gap-2">
                        {aboutUs.vision.title}
                        <motion.span
                          className="inline-block w-2 h-2 rounded-full bg-blue-600"
                          animate={{ scale: [1, 1.5, 1] }}
                          transition={{ duration: 2, repeat: Infinity }}
                        />
                      </h3>
                      <p className="text-lg text-slate-600 leading-relaxed">{aboutUs.vision.text}</p>
                    </div>
                  </div>
                  
                  {/* Decorative corner */}
                  <div className="absolute top-4 right-4 w-16 h-16 border-t-2 border-r-2 border-blue-200 rounded-tr-2xl opacity-50" />
                </div>
              </motion.div>

              {/* Mission */}
              <motion.div
                className="relative group"
                whileHover={{ scale: 1.02 }}
                transition={{ type: "spring", stiffness: 300 }}
              >
                <div className="absolute inset-0 bg-gradient-to-br from-purple-600/20 to-pink-600/20 rounded-3xl blur-xl opacity-0 group-hover:opacity-100 transition-opacity duration-500" />
                <div className="relative bg-white/80 backdrop-blur-sm rounded-3xl p-8 shadow-xl border border-slate-200/50 group-hover:border-purple-300/50 transition-all duration-500">
                  <div className="flex items-start gap-4">
                    <motion.div
                      className="flex-shrink-0 w-14 h-14 rounded-2xl bg-gradient-to-br from-purple-600 to-pink-600 flex items-center justify-center shadow-lg"
                      whileHover={{ rotate: [0, -10, 10, -10, 0] }}
                      transition={{ duration: 0.5 }}
                    >
                      <Compass className="w-7 h-7 text-white" />
                    </motion.div>
                    <div className="flex-1">
                      <h3 className="text-2xl font-bold text-slate-900 mb-3 flex items-center gap-2">
                        {aboutUs.mission.title}
                        <motion.span
                          className="inline-block w-2 h-2 rounded-full bg-purple-600"
                          animate={{ scale: [1, 1.5, 1] }}
                          transition={{ duration: 2, repeat: Infinity, delay: 0.5 }}
                        />
                      </h3>
                      <p className="text-lg text-slate-600 leading-relaxed">{aboutUs.mission.text}</p>
                    </div>
                  </div>
                  
                  {/* Decorative corner */}
                  <div className="absolute bottom-4 left-4 w-16 h-16 border-b-2 border-l-2 border-purple-200 rounded-bl-2xl opacity-50" />
                </div>
              </motion.div>
            </motion.div>

            {/* Values Section */}
            <motion.div 
              className="space-y-8"
              initial={{ opacity: 0, x: 30 }}
              whileInView={{ opacity: 1, x: 0 }}
              viewport={{ once: true }}
              transition={{ duration: 0.7, delay: 0.2 }}
            >
              <div className="text-center md:text-left">
                <h3 className="text-2xl font-bold text-slate-900 mb-2 inline-flex items-center gap-2">
                  {aboutUs.values.title}
                  <motion.div
                    animate={{ rotate: 360 }}
                    transition={{ duration: 3, repeat: Infinity, ease: "linear" }}
                  >
                    <Sparkles className="w-5 h-5 text-amber-500" />
                  </motion.div>
                </h3>
              </div>

              <div className="grid grid-cols-1 gap-6 sm:grid-cols-2">
                {aboutUs.values.items.map((value, index) => {
                  const config = valueConfig[value.title];
                  const Icon = config.icon;
                  const isHovered = hoveredValue === index;

                  return (
                    <motion.div
                      key={value.title}
                      initial={{ opacity: 0, scale: 0.8, rotateY: -15 }}
                      whileInView={{ opacity: 1, scale: 1, rotateY: 0 }}
                      viewport={{ once: true }}
                      transition={{ 
                        duration: 0.6, 
                        delay: index * 0.1,
                        type: "spring",
                        stiffness: 100
                      }}
                      onMouseEnter={() => setHoveredValue(index)}
                      onMouseLeave={() => setHoveredValue(null)}
                      className="relative group perspective-1000"
                    >
                      {/* Glow Effect */}
                      <motion.div
                        className={`absolute -inset-1 bg-gradient-to-r ${config.gradient} rounded-3xl blur-xl opacity-0 group-hover:opacity-30 transition-opacity duration-500`}
                        animate={isHovered ? { scale: [1, 1.05, 1] } : {}}
                        transition={{ duration: 2, repeat: Infinity }}
                      />

                      {/* Card */}
                      <motion.div
                        className={`relative h-72 rounded-3xl overflow-hidden shadow-2xl transition-all duration-500 border-2 border-transparent group-hover:border-white/50`}
                        whileHover={{ 
                          y: -8,
                          rotateX: 5,
                          rotateY: 5,
                        }}
                        transition={{ type: "spring", stiffness: 300 }}
                      >
                        {/* Background with Gradient */}
                        <div className={`absolute inset-0 bg-gradient-to-br ${config.bgGradient} transition-all duration-500`} />
                        
                        {/* Animated Pattern Overlay */}
                        <motion.div
                          className="absolute inset-0 opacity-10"
                          style={{
                            backgroundImage: `radial-gradient(circle at 50% 50%, currentColor 1px, transparent 1px)`,
                            backgroundSize: '20px 20px',
                          }}
                          animate={{
                            backgroundPosition: ['0px 0px', '20px 20px'],
                          }}
                          transition={{ duration: 20, repeat: Infinity, ease: "linear" }}
                        />

                        {/* Gradient Overlay */}
                        <motion.div
                          className={`absolute inset-0 bg-gradient-to-br ${config.gradient} opacity-0 group-hover:opacity-20 transition-opacity duration-500`}
                        />

                        {/* Content */}
                        <div className="relative h-full flex flex-col justify-between p-6 z-10">
                          {/* Icon */}
                          <motion.div
                            className={`w-16 h-16 rounded-2xl bg-gradient-to-br ${config.gradient} flex items-center justify-center shadow-xl`}
                            whileHover={{ 
                              scale: 1.1,
                              rotate: [0, -5, 5, -5, 0]
                            }}
                            transition={{ duration: 0.5 }}
                          >
                            <Icon className="w-8 h-8 text-white" />
                          </motion.div>

                          {/* Text Content */}
                          <div>
                            <h4 className="text-xl font-bold text-slate-900 mb-2 drop-shadow-sm">
                              {value.title}
                            </h4>
                            <motion.p
                              className="text-sm text-slate-700 leading-relaxed"
                              initial={{ opacity: 0.7 }}
                              animate={{ opacity: isHovered ? 1 : 0.7 }}
                              transition={{ duration: 0.3 }}
                            >
                              {value.text}
                            </motion.p>
                          </div>

                          {/* Floating particles */}
                          {isHovered && (
                            <>
                              {[...Array(3)].map((_, i) => (
                                <motion.div
                                  key={i}
                                  className={`absolute w-2 h-2 rounded-full bg-gradient-to-r ${config.gradient}`}
                                  initial={{
                                    x: Math.random() * 200 - 100,
                                    y: Math.random() * 200 - 100,
                                    opacity: 0,
                                    scale: 0,
                                  }}
                                  animate={{
                                    y: [0, -100],
                                    opacity: [0, 1, 0],
                                    scale: [0, 1, 0],
                                  }}
                                  transition={{
                                    duration: 2,
                                    delay: i * 0.2,
                                    repeat: Infinity,
                                  }}
                                  style={{
                                    left: '50%',
                                    top: '50%',
                                  }}
                                />
                              ))}
                            </>
                          )}
                        </div>

                        {/* Corner Accent */}
                        <motion.div
                          className="absolute bottom-0 right-0 w-24 h-24 opacity-20"
                          animate={isHovered ? { rotate: 360 } : {}}
                          transition={{ duration: 3, ease: "linear" }}
                        >
                          <div className={`absolute bottom-0 right-0 w-full h-full bg-gradient-to-tl ${config.gradient} rounded-tl-full`} />
                        </motion.div>
                      </motion.div>
                    </motion.div>
                  );
                })}
              </div>
            </motion.div>
          </div>
        </div>
      </div>
    </section>
  );
}